/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "rf_system_1channel.h"
#include <QCoreApplication>
#include <QDebug>
#include <QMessageBox>

#ifdef ETHERCAT
	RF_System_1channel::RF_System_1channel() : ethercatModule(*this)
#else
	RF_System_1channel::RF_System_1channel()
#endif
{
	qDebug() << "RF_System_1channel: [Reading config file]";
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	/* Modbus TCP Preparation */
	mbServer = new ModbusServer(QCoreApplication::applicationDirPath() + "/modbus.txt");
	mbServer->Initialize();

	for (int i = 0; i < max_channel_count; i++)
	{
		channels_working_state.append(false);
		PSU_readings_valid_state.append(false);
	}

	/* GPIO preparation for Mini-Circuits Raspberry Pi HAT */
#if defined(Q_OS_LINUX)

	if (config->get_support_HAT_B14_0835())
	{
		GPIO_control_external = new GPIOClass("6");
		GPIO_control_SGx = new GPIOClass("12");
		GPIO_LED_1 = new GPIOClass(QString::number(13));

		GPIO_control_external->setDirection("out");
		GPIO_control_SGx->setDirection("out");
		GPIO_LED_1->setDirection("out");

		/* Configure default state for the Amplifier Mode GPIOs */
		set_GPIO_control_SGx(1);
		set_GPIO_control_external(0);

		/* Connect the LED timer to blink the LED if necessary */
		timer_LED = new QTimer(this);			//Timer for LED blinking
		connect(timer_LED, &QTimer::timeout, this, &RF_System_1channel::LED_toggle);
	}


	/* Optional GPIO-based PSU hardware enable button + interlock loop */
	if (config->get_GPIO_interlock_in() >= 0)
	{
		GPIO_interlock_in = new GPIOClass(QString::number(config->get_GPIO_interlock_in()));
		GPIO_interlock_in->setDirection("in");
	}

	if (config->get_GPIO_3v3_out() >= 0)
	{
		GPIO_3v3_out = new GPIOClass(QString::number(config->get_GPIO_3v3_out()));		//Provides 3.3V that goes through Front Panel button + Interlocks switches and is read back by GPIO 19.
		GPIO_3v3_out->setDirection("out");

		/* Set the GPIO high. */
		GPIO_3v3_out->setValue("1");
	}


	/* Timer for GPIO related actions */
	timer_actions = new QTimer();
	connect(timer_actions, &QTimer::timeout, this, &RF_System_1channel::timer_actions_slot);

	timer_actions->start(200);

#endif

	threadList = new QList<QThread*>();

	/* Pick a COM port
	 * Initialize RF channels
	 * Connect signals and slots
	 * Move RF Channels to their own Threads */
	for (int i = 0; i < max_channel_count; i++)
	{
		/* Choose serial port for the RF Channel */
		QList<QSerialPort*> portlist = get_port_list();
		QString portname = "";

		/* Select the first port from the list, or if the list is empty show an error message. */
		if (!portlist.isEmpty())
		{
			if (portlist.size() > 1)
			{
				qDebug() << "RF_System_1channel: Multiple supported devices detected.";
			}
			portname = portlist.at(0)->portName();
		}
		else
		{
			QMessageBox message;
			message.critical(nullptr,	"No Device Found",
							 "No supported devices were found."
							 "\nCannot establish a connection.\n"
							 "\nPlease try the following:"
							 "\n1. Ensure a supported device is connected."
							 "\n2. Restart this application to try again."
							 "\n3. If the problem persist, try manually specifying a port in the configuration file.");
		}

		qDebug() << "RF_System_1channel: Channel" << i+1 << "connecting to" << portname;

		/* Create the RF channel with the chosen serial port */
		channel = new RF_Channel(i+1, i+1, portname);
		RF_System::Channels->append(channel);

		#if defined(Q_OS_LINUX)
		if (config->get_support_HAT_B14_0835())
		{
			//Amplifier mode
			connect(RF_System::Channels->at(i), &RF_Channel::signal_GPIO_amplifier_mode_enabled_set, this, &RF_System_1channel::set_GPIO_amplifier_mode_enabled);
			connect(this, &RF_System_1channel::signal_GPIO_amplifier_mode_get, RF_System::Channels->at(i), &RF_Channel::handler_GPIO_amplifier_mode_get);

			//LED management
			qRegisterMetaType<LED_MODE>();		//Necessary for cross-thread signal and slot communication.
			connect(RF_System::Channels->at(i), &RF_Channel::signal_LED_mode_set, this, &RF_System_1channel::handler_LED_mode_set);
		}

		//PSU enable switch
		if (config->get_GPIO_interlock_in() >= 1)
		{
			connect(this, &RF_System_1channel::signal_GPIO_PSU_button_enable_get, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Enable_Safely);
		}
		#endif

		//Channel Working
		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_working, this, &RF_System_1channel::handler_channel_working);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_readings_valid, this, &RF_System_1channel::handler_PSU_readings_valid);

		//
		// TODO:
		// All this Modbus stuff shouldn't even get connected if Modbus isn't supported according to the config file.
		//

		/* Modbus Setup */
//		connect(this, &RF_System_1channel::signal_set_generator_ready, mbServer, &ModbusServer::handler_generator_ready_get);
		connect(this, &RF_System_1channel::signal_SGx_communication_working, mbServer, &ModbusServer::handler_SGx_communication_working_get);
		connect(this, &RF_System_1channel::signal_PSU_communication_working, mbServer, &ModbusServer::handler_PSU_communication_working_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_RF_enable_get, mbServer, &ModbusServer::handler_RF_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DLL_enable_get, mbServer, &ModbusServer::handler_DLL_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PWM_settings_get, mbServer, &ModbusServer::handler_PWM_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_power_get, mbServer, &ModbusServer::handler_power_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PA_power_readings, mbServer, &ModbusServer::handler_PA_power_readings);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_frequency_get, mbServer, &ModbusServer::handler_frequency_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_clock_source_get, mbServer, &ModbusServer::handler_get_clock_source);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_SWP_measurements_get, mbServer, &ModbusServer::handler_SWP_measurement_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_SWP_settings_get, mbServer, &ModbusServer::handler_SWP_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_temperature_PA_get, mbServer, &ModbusServer::handler_temperature_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_error_get, mbServer, &ModbusServer::handler_error_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_IU_reading_get, mbServer, &ModbusServer::handler_PSU_IU_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_enable_combined_get, mbServer, &ModbusServer::handler_PSU_enable_combined_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_power_efficiency_get, mbServer, &ModbusServer::handler_PSU_power_efficiency_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_dissipation_get, mbServer, &ModbusServer::handler_PSU_dissipation_get);

		//ODC WORKAROUND!!
//		connect(mbServer, &ModbusServer::signal_set_RF_enable, this, &RF_System_1channel::set_RF_enable_workaround);
//		connect(mbServer, &ModbusServer::signal_execute_sweep, this, &RF_System_1channel::execute_sweep_dBm_workaround);
//		connect(this, &RF_System_1channel::signal_set_power_control_mode, RF_System::Channels->at(i), &RF_Channel::Set_Power_Control_Mode);
//		connect(this, &RF_System_1channel::signal_execute_sweep_2, RF_System::Channels->at(i), &RF_Channel::Execute_SWP_dBm2);
//		connect(this, &RF_System_1channel::signal_set_RF_enable, RF_System::Channels->at(i), &RF_Channel::Set_RF_enable);			//This one already exists

		connect(mbServer, &ModbusServer::signal_set_RF_enable, RF_System::Channels->at(i), &RF_Channel::Set_RF_enable);
		connect(mbServer, &ModbusServer::signal_execute_reset_SGx, RF_System::Channels->at(i), &RF_Channel::Execute_Reset_SGx);
		connect(mbServer, &ModbusServer::signal_execute_reset_protection, RF_System::Channels->at(i), &RF_Channel::Execute_Reset_Protection);
		connect(mbServer, &ModbusServer::signal_execute_error_clear, RF_System::Channels->at(i), &RF_Channel::Execute_Error_Clear);
		connect(mbServer, &ModbusServer::signal_execute_sweep, RF_System::Channels->at(i), &RF_Channel::Execute_SWP_dBm2);
		connect(mbServer, &ModbusServer::signal_set_DLL_enable, RF_System::Channels->at(i), &RF_Channel::Set_DLL_enable);
		connect(mbServer, &ModbusServer::signal_set_PWM_enable, RF_System::Channels->at(i), &RF_Channel::Set_PWM_enable);
//		connect(mbServer, &ModbusServer::signal_PSU_interlock, this, &RF_System_1channel::handler_PSU_interlock_modbus);	//update enable, voltage setpoint & current limit.
		connect(mbServer, &ModbusServer::signal_set_power_control_mode, RF_System::Channels->at(i), &RF_Channel::Set_Power_Control_Mode);

		connect(mbServer, &ModbusServer::signal_set_power_watt, RF_System::Channels->at(i), &RF_Channel::Set_Power_Watt);
		connect(mbServer, &ModbusServer::signal_set_frequency, RF_System::Channels->at(i), &RF_Channel::Set_Frequency);
		connect(mbServer, &ModbusServer::signal_set_SWP_frequency_start, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Start);
		connect(mbServer, &ModbusServer::signal_set_SWP_frequency_stop, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Stop);
		connect(mbServer, &ModbusServer::signal_set_SWP_frequency_step, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Step);
		connect(mbServer, &ModbusServer::signal_set_SWP_power, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Power_Watt);
		connect(mbServer, &ModbusServer::signal_set_DLL_frequency_limit_lower, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Lower_Limit);
		connect(mbServer, &ModbusServer::signal_set_DLL_frequency_limit_upper, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Upper_Limit);
		connect(mbServer, &ModbusServer::signal_set_DLL_frequency_start, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Start);
		connect(mbServer, &ModbusServer::signal_set_DLL_frequency_step, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Step);
		connect(mbServer, &ModbusServer::signal_set_DLL_threshold, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Threshold);
		connect(mbServer, &ModbusServer::signal_set_DLL_delay, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Main_Delay);
		connect(mbServer, &ModbusServer::signal_set_PWM_frequency, RF_System::Channels->at(i), &RF_Channel::Set_PWM_Frequency);
		connect(mbServer, &ModbusServer::signal_set_PWM_duty_cycle, RF_System::Channels->at(i), &RF_Channel::Set_PWM_Duty_Cycle_Target);
		connect(mbServer, &ModbusServer::signal_set_PID_Kp, RF_System::Channels->at(i), &RF_Channel::Set_PID_Kp);
		connect(mbServer, &ModbusServer::signal_set_PID_Ki, RF_System::Channels->at(i), &RF_Channel::Set_PID_Ki);
		connect(mbServer, &ModbusServer::signal_set_PID_Kd, RF_System::Channels->at(i), &RF_Channel::Set_PID_Kd);
		connect(mbServer, &ModbusServer::signal_set_PID_setpoint, RF_System::Channels->at(i), &RF_Channel::Set_PID_Setpoint);
		connect(mbServer, &ModbusServer::signal_set_PID_scaling, RF_System::Channels->at(i), &RF_Channel::Set_PID_Scaling);
		connect(mbServer, &ModbusServer::signal_set_PID_offset, RF_System::Channels->at(i), &RF_Channel::Set_PID_Offset);
		connect(mbServer, &ModbusServer::signal_set_clock_source, RF_System::Channels->at(i), &RF_Channel::Set_Clock_Source);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_phase_get, mbServer, &ModbusServer::handler_get_phase);
		connect(mbServer, &ModbusServer::signal_set_phase, RF_System::Channels->at(i), &RF_Channel::Set_Phase);

		/* Avoid unnecessary setRegister errors (due to missing register definitions for phase items) if phase setting is disabled anyway */
//		if (config->get_support_phase_control() == true)
//		{
//			/* $PCS is used for both SGx phase and RF splitter channel phases
//			 * Only one of these should be active at a time */
//			if (config->get_phase_gain_board_mode() == 0)
//			{
//				connect(RF_System::Channels->at(i), &RF_Channel::signal_phase_get, mbServer, &ModbusServer::handler_get_phase);
//				connect(mbServer, &ModbusServer::signal_set_phase, RF_System::Channels->at(i), &RF_Channel::Set_Phase);
//			}
//			else if (config->get_phase_gain_board_mode() == 1)
//			{
//				connect(mbServer, &ModbusServer::signal_set_PGB_phase, RF_System::Channels->at(i), &RF_Channel::Set_PGB_Phase);
//				connect(RF_System::Channels->at(i), &RF_Channel::signal_PGB_phase_get, mbServer, &ModbusServer::handler_get_PGB_phase);
//			}
//		}

		//Thread management
		channelThread = new QThread();
		threadList->append(channelThread);

		connect(channelThread, &QThread::finished, channel, &QObject::deleteLater);
		channel->moveToThread(channelThread);

		channelThread->start();
	}

	//
	// TODO:
	// This should probably be moved into GUI class
	//
	Remote_Command_Mode = new RCM_Class(*RF_System::Channels);
	connect(Remote_Command_Mode, &RCM_Class::signal_RCM_mode, this, &RF_System_1channel::handler_remote_command_mode_get);
}

RF_System_1channel::~RF_System_1channel()
{
	for (int i = 0; i < max_channel_count; i++)
	{
		threadList->at(i)->quit();
		threadList->at(i)->wait();
	}

#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835() == true)
	{
		GPIO_LED_1->setValue("0");
		timer_LED->stop();
	}
#endif

}

#if defined(Q_OS_LINUX)
void RF_System_1channel::timer_actions_slot()
{
	if (config->get_support_HAT_B14_0835())
	{
		poll_GPIO_amplifier_mode();
	}

	/* if a GPIO is provided for interlock functionality, then poll it */
	if 	(config->get_GPIO_interlock_in() >= 0)
	{
		poll_GPIO_interlock_in();
	}
}
#endif


/*
 * Find a serial port to use:
 * If config file contains a defined port --> Use the defined port.
 * If there is no defined port --> Try auto-detecting a supported device based on the Vendor ID and Product ID.
 *
 * Supported Devices:
 * Vendor ID: 0x1fc9 = 8137 && Product ID: 0x0083 = 131
 *
 * Remark: this method may return false positives if there are other devices plugged in that use the same USB driver.
 */
QList<QSerialPort*> RF_System_1channel::get_port_list()
{
	QSerialPort *port;
	QList<QSerialPort*> list_of_ports;
	list_of_ports.clear();

	if (config->get_SGX_port(1) != "")
	{
		port = new QSerialPort(config->get_SGX_port(1));
		qDebug() << "Specified Serial Port:" << port->portName();
		list_of_ports.append(port);
	}
	else
	{
		QList<QSerialPortInfo> infolist;
		for (const QSerialPortInfo& info : QSerialPortInfo::availablePorts())
		{
			if ((info.vendorIdentifier() == 8137 && info.productIdentifier() == 131))
			{
				qDebug() << "Valid device detected at port:" << info.portName();
				port = new QSerialPort(info);
				list_of_ports.append(port);
			}
		}
	}

	return list_of_ports;
}


//#error TODO - probe the device behind the port to ascertain it's the same one!

/**********************************************************************************************************************************************************************************
 * GPIO interactions
 *********************************************************************************************************************************************************************************/
/* GPIO Set/Get interactions should be handled in the RF system class.
 * There can theoretically be multiple RF channels within a single RF System, and you probably shouldn't have them all setting and reading the same GPIO. */

/* GPIOs - Amplifier mode  */

/* Poll the external control and SGx control GPIOs for the 1kW */
#if defined(Q_OS_LINUX)
void RF_System_1channel::poll_GPIO_amplifier_mode()
{
	if (config->get_support_HAT_B14_0835())
	{
		int mode = determine_GPIO_amplifier_mode();
		if (mode >= 0 && mode <= 1)
		{
			emit signal_GPIO_amplifier_mode_get(1, (bool)mode);
		}
	}
}

int RF_System_1channel::determine_GPIO_amplifier_mode()
{
	int GPIO_amplifier_mode = -1;	//default invalid

	if (config->get_support_HAT_B14_0835())
	{
		int control_SGx = get_GPIO_control_SGx();
		int control_external = get_GPIO_control_external();

		if (control_SGx >= 1 && control_external == 0)
		{
			GPIO_amplifier_mode = 0;
		}
		else if (control_SGx == 0 && control_external >= 1)
		{
			GPIO_amplifier_mode = 1;
		}
		else
		{
			qDebug() << "RF System 1channel: GPIO Amplifier Mode enable cannot be determined" << control_SGx << control_external;
		}
	}
	return GPIO_amplifier_mode;
}

void RF_System_1channel::set_GPIO_amplifier_mode_enabled(int intrasys_num, bool enable)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	if (config->get_support_HAT_B14_0835())
	{
		if (enable == true)
		{
			set_GPIO_control_SGx(0);
			set_GPIO_control_external(1);
		}
		else
		{
			set_GPIO_control_SGx(1);
			set_GPIO_control_external(0);
		}

		poll_GPIO_amplifier_mode();
	}
}

int RF_System_1channel::get_GPIO_control_SGx()
{
	if (config->get_support_HAT_B14_0835())
	{
		return GPIO_control_SGx->readValue();
	}

	return -1;
}

void RF_System_1channel::set_GPIO_control_SGx(int value)
{
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_control_SGx->setValue(QString::number(value));
	}
}

int RF_System_1channel::get_GPIO_control_external()
{
	if (config->get_support_HAT_B14_0835())
	{
		return GPIO_control_external->readValue();
	}

	return -1;
}

void RF_System_1channel::set_GPIO_control_external(int value)
{
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_control_external->setValue(QString::number(value));
	}
}


/* GPIOs - PSU enable button */

/* Poll the external control and SGx control GPIOs for the 1kW */
void RF_System_1channel::poll_GPIO_interlock_in()
{
	if (config->get_GPIO_interlock_in())
	{
		int GPIO_interlock_in_state = GPIO_interlock_in->readValue();

		/* Front panel button state changed */
		if (last_GPIO_interlock_in_state != GPIO_interlock_in_state)
		{
//			GPIO_PSU_timer->start(1000);
			handler_GPIO_PSU_timer();
		}

		last_GPIO_interlock_in_state = GPIO_interlock_in_state;
	}
}

void RF_System_1channel::handler_GPIO_PSU_timer()
{
	/* Check the button state again, to ensure the PSU is really enabled and it wasn't just a false positive. */
	int GPIO_state = GPIO_interlock_in->readValue();
	if (GPIO_state == -1)
	{
		return;
	}
	else
	{
		emit signal_GPIO_PSU_button_enable_get(1, (bool)GPIO_state);
	}
}

/* GPIO - LED handling */

/*
 * Handle LED lighting
 * Mode 0 - RF OFF – LED OFF
 * Mode 1 - RF ON – LED ON (solid)
 * Mode 2 - ALARM - LED Blink (500ms ON -> 500ms OFF)
 * Mode 3 - Blind RCM - LED Blink (1000ms ON -> 1000ms OFF)
 */
void RF_System_1channel::handler_LED_mode_set(int intrasys_num, LED_MODE mode)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	led_mode = mode;
	configure_LED_mode();
}

void RF_System_1channel::configure_LED_mode()
{
	if (config->get_support_HAT_B14_0835() == false)
	{
		return;
	}

	LED_MODE mode = led_mode;

	/* if RCM blind is enabled -> LED should do slow blink, otherwise go with whatever RF Channel says */
	if (rcm_mode == RCM_MODE::RCM_MODE_USB_BLIND ||
		rcm_mode == RCM_MODE::RCM_MODE_TCP_BLIND ||
		rcm_mode == RCM_MODE::RCM_MODE_RS232_BLIND ||
		rcm_mode == RCM_MODE::RCM_MODE_RS485_BLIND
		)
	{
		mode = LED_MODE::LED_MODE_RCM;
	}

	/* Configure the LED behaviour */
	switch (mode)
	{
		case LED_MODE::LED_MODE_OFF:
			GPIO_LED_1->setValue("0");
			timer_LED->stop();
			break;
		case LED_MODE::LED_MODE_ON:
			GPIO_LED_1->setValue("1");
			timer_LED->stop();
			break;
		case LED_MODE::LED_MODE_ALARM:
			GPIO_LED_1->setValue("1");  //flip ON for the error state
			timer_LED->start(500);		//Fast blink for errors
			break;
		case LED_MODE::LED_MODE_RCM:
			GPIO_LED_1->setValue("1");  //flip ON for the error state
			timer_LED->start(1000);		//Slow blink for RCM
			break;
		default:
			GPIO_LED_1->setValue("0");
			timer_LED->stop();
			break;
	}
}

/* Invert the GPIO_LED value every time this function is called */
void RF_System_1channel::LED_toggle()
{
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_LED_1->setValue(QString::number(!(bool)GPIO_LED_1->readValue()));
	}
}


#endif

/**********************************************************************************************************************************************************************************
 * Remote Command Mode
 *********************************************************************************************************************************************************************************/
void RF_System_1channel::handler_remote_command_mode_get(RCM_MODE mode)
{
	rcm_mode = mode;

#if defined (Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		configure_LED_mode();
	}
#endif
}

/**********************************************************************************************************************************************************************************
 * Generator Ready: Tell modbus that Generator is ready for use
 *********************************************************************************************************************************************************************************/
//Check if all channels are working; each channel updates this function individually, but verdict is based on the combined results.
void RF_System_1channel::handler_channel_working(int intrasys_num, bool enabled)
{
	channels_working_state[intrasys_num-1] = enabled;

	/* if any single channel returns a 'false' working state, flip the channels_working state to false, else true. */
	channels_working = true;
	for (int i = 1; i <= max_channel_count; i++)
	{
		if (channels_working_state[i-1] == false)
		{
			channels_working = false;
			break;
		}
	}

	emit signal_SGx_communication_working(channels_working);

	/* If a channel recovers, set the last interlock GPIO state to -1 (invalid), so that the next time it's read back, both the PSU and the RF splitter get reconfigured to the desired state */
#if defined (Q_OS_LINUX)
	if (config->get_GPIO_interlock_in() >= 1)
	{
		if (channels_working == true)
		{
			qDebug() << "RF_System_1channel::handler_channel_working -> resetting the 'interlock in' GPIO state";
			last_GPIO_interlock_in_state = -1;
		}
	}
#endif
}

/* Exist to communicate that despite PSUDG returning values without an error, the values are invalid.
 * Signal sends valid 'false' if the voltage or current of a PSU are less than 0.
 * For example due to RF noise sabotaging the measurements themselves the data travelling over the I2C bus. */
void RF_System_1channel::handler_PSU_readings_valid(int intrasys_num, bool valid)
{
	PSU_readings_valid_state[intrasys_num-1] = valid;

	// if any channel returns false working state, flip the channels working state to false, else true.
	PSUs_readings_valid = true;
	for (int i = 1; i <= max_channel_count; i++)
	{
		if (PSU_readings_valid_state[i-1] == false)
		{
			PSUs_readings_valid = false;
			break;
		}
	}

	emit signal_PSU_communication_working(PSUs_readings_valid);
}
